import hashlib
import hmac
import json
from urllib.parse import parse_qsl
from typing import Optional, Dict, Any

from werkzeug.security import generate_password_hash, check_password_hash

def hash_password(pw: str) -> str:
    return generate_password_hash(pw)

def verify_password(pw: str, pw_hash: str) -> bool:
    return check_password_hash(pw_hash, pw)

def validate_webapp_init_data(init_data: str, bot_token: str) -> Optional[Dict[str, Any]]:
    """
    Returns parsed payload (including 'user') if valid, else None.
    """
    if not init_data:
        return None

    pairs = list(parse_qsl(init_data, keep_blank_values=True))
    data = dict(pairs)
    received_hash = data.get("hash")
    if not received_hash:
        return None

    data_check_items = []
    for k, v in pairs:
        if k == "hash":
            continue
        data_check_items.append(f"{k}={v}")
    data_check_items.sort()
    data_check_string = "\n".join(data_check_items)

    secret_key = hashlib.sha256(bot_token.encode("utf-8")).digest()
    computed_hash = hmac.new(secret_key, data_check_string.encode("utf-8"), hashlib.sha256).hexdigest()

    if not hmac.compare_digest(computed_hash, received_hash):
        return None

    user_raw = data.get("user")
    if user_raw:
        try:
            data["user"] = json.loads(user_raw)
        except Exception:
            data["user"] = None

    return data